LINK_DOWN = 0
LINK_UP = 1

# Set this to the string which identifies that a packet send/recv failed in RON
# (this was 'NULL' in the first RON traces, and '0' in the latest traces)
NULL = '0'


##
## Topology map based on RON traces
##
class Topology:

    # Initialize, read start time from trace, reset global clock
    # Meaning of the ron variable:
    #     '0' = direct on the Internet (default setting)
    #     '1' = latency optimized RON path
    #     '2' = loss optimized RON path
    def __init__(self, tracefile, ron='0', debug=0):
        self.tracefile = tracefile
        self.hosts = {}
        self.clock = 0.0
        self.traceclock = 0.0
        self.ron = ron
        self.downlinks = 0
        self.debug = debug
        self.trace_clock = 0.0
        # Use first epoch value in trace as virtual time 0
        self.start = float(self.tracefile.readline().split('\t')[3])
        self.tracefile.seek(0,0)

    # Advance global clock with 'tick' seconds and update link info
    def advance(self, tick):
        # Only advance to next timeslot if the current tick advances the current
        # timeslot from the trace file
        if self.trace_clock > self.clock + tick:
            self.clock += tick
            return

        while 1:
            # Read probe data one line at a time
            line = self.tracefile.readline()
            if not line:
                print 'Trace file ended before timeslice', self.clock + tick
                return 0
            line = line.rstrip()
            src, dst, ron, s1, r1, s2, r2 = line.split('\t')

            if self.ron == ron:
                # Update uplink
                if not self.hosts.has_key(src):
                    self.hosts[src] = {}
                if r1 == NULL:
                    self.hosts[src][dst] = LINK_DOWN
                    self.downlinks += 1
                    if self.debug:
                        print 'Down: %s -> %s @ %.1f' % (src, dst, float(s1)-self.start)
                else:
                    self.hosts[src][dst] = LINK_UP

                # Update backlink
                if not self.hosts.has_key(dst):
                    self.hosts[dst] = {}
                if r2 == NULL:
                    self.hosts[dst][src] = LINK_DOWN
                    self.downlinks += 1
                    if self.debug:
                        print 'Down: %s <- %s @ %.1f' % (src, dst, float(s1)-self.start)
                else:
                    self.hosts[dst][src] = LINK_UP

            # Check if we have used our timeslice, and if so return
            current_clock = float(s1) - self.start
            if current_clock >= self.clock + tick:
                self.clock = self.clock + tick
                self.trace_clock = current_clock
                return 1

    # Return true if connectivity from src->dst is up, false otherwise
    def link(self, src, dst):
        try:
            return self.hosts[src][dst]
        except:
            return 0

## Commandline arguments
if __name__=="__main__":
    import sys, pprint

    if len(sys.argv) < 2:
        print "Error: supply RON latency file as parameter" 
        raise SystemExit

    t = Topology(open(sys.argv[1]), ron='0', debug=1)
    for a in range(2000):
        if t.advance(1) == 0:
            break

    pprint.pprint(t.hosts)
    print
    print 'Total number of downlinks during %d seconds: %d ' % \
          (float(t.clock), t.downlinks)
    print 'Total number of hosts: ', len(t.hosts)
    print 'List of hosts found in trace:'
    for h in t.hosts:
        print h

