# Caveats: Lots of duplicate entries are generated.  The time starts at 0 and not
# the actual time the traces were done (this cannot be automatically established anyway
# so we should perhaps just make this is into a commandline argument).

import sys, os, glob, marshal
import ping

# This is for the second trace we got from omar
host_mappings = {
    'MIT':'goose.lcs.mit.edu',
    'MASS-ISP1':'mediaone-ma.ron.lcs.mit.edu',
    'University-of-Utah':'utah.ron.lcs.mit.edu',
    'SightPath':'sightpath.ron.lcs.mit.edu',
    'UTAH-ISP1':'ccicom.ron.lcs.mit.edu',
    'Cornell':'cornell.ron.lcs.mit.edu',
    'CAL-ISP1':'msanders.ron.lcs.mit.edu',
    'UTAH-ISP3':'aros.ron.lcs.mit.edu',
    'NYU':'nyu.ron.lcs.mit.edu',
    'Netherlands':'nl.ron.lcs.mit.edu',
    'CMU':'cmu.ron.lcs.mit.edu',
    'Sweden-Lulea':'lulea.ron.lcs.mit.edu',
    'MASS-ISP2':'mazu1.ron.lcs.mit.edu',
    'CAL-ISP2':'pdi.ron.lcs.mit.edu',
    'NC-ISP':'nc.ron.lcs.mit.edu',
    'Korea':'kr.ron.lcs.mit.edu',
    'Greece':'gr.ron.lcs.mit.edu',
    'UTAH-ISP2':'cheetah.4b4.net',
    'UCSD':'pisa.ucsd.edu',
    'Taiwan':'jane.im.ntu.edu.tw',
    'Spain':'asterix.fi.upm.es',
    'Technion':'tochna12.technion.ac.il',
    'Sweden-Stockholm':'tcs32.nada.kth.se',
    'Italy':'www.security.unisa.it',
    'Australia':'flexal.cs.usyd.edu.au',
    'New-Zealand':'st-james.mcs.vuw.ac.nz',
    'Hebrew-University':'gw.cs.huji.ac.il',
    'Turkey':'external.ceng.metu.edu.tr',
    'Portugal':'dream.cc.fc.ul.pt',
    'Switzerland':'lpdpc1.epfl.ch',
    'Brazil':'www.ufpb.br'
}

class Ping2Ron:

    def __init__(self, dir, max_time):
        self.dir = dir
        self.hosts = host_mappings.keys()
        os.chdir(dir)
        self.links = {}
        self.max_time = max_time

    ## Taverse all the ping files using the ping tracefile parser (from ping.py)
    def traverse(self):
        for host in self.hosts:
            sys.stderr.write('\nTraversing ping trace recorded at host %s: ' % host)
            sys.stderr.flush()
            target_hosts = glob.glob(os.path.join(host,"*"))
            if len(target_hosts) == 0:
                sys.stderr.write('no ping traces found, ignoring host')
                sys.stderr.flush()
                continue
            self.links[host_mappings[host]] = {}
            for t in target_hosts:
                t_host = os.path.basename(t)
                pingfile = os.path.join(t, 'ping.stat')
                p = ping.Ping(open(pingfile))
                self.links[host_mappings[host]][t_host] = p.traverse_file()
                sys.stderr.write('.')
                sys.stderr.flush()
            
    ## Output a RON compatible trace file based on the ping trace files
    def output_ron(self):
        assert len(self.links) > 0, "No links to output"
        assert self.max_time > 0, "Must output trace logs for more than 0 seconds"

        sys.stderr.write('Starting to output RON trace file, this may take a while...\n')
        sys.stderr.flush()
        for ts in range(0, self.max_time+60, 60):
            for s_host in self.links.keys():
                for t_host in self.links[s_host].keys():
                    # Ignore hosts without symmetric communication patterns (like NL in experiment 1)
                    if not self.links.has_key(t_host):  continue
                    sys.stdout.write("%s\t%s\t0\t%.1f\t" % (s_host, t_host, ts))
                    if not self.links[s_host].has_key(t_host) or\
                       self.links[s_host][t_host].has_key(ts) or\
                       ts >= self.links[s_host][t_host][ping.END_MARKER]:
                        sys.stdout.write("NULL\tNULL\tNULL\n")
                    else:
                        sys.stdout.write("%.1f\t%.1f\t" % (ts, ts))
                        if not self.links[t_host].has_key(s_host) or self.links[t_host][s_host].has_key(ts):
                            sys.stdout.write("NULL\n")
                        else:
                            sys.stdout.write("%.1f\n" % (ts))

        sys.stderr.write('Finished\n')
        sys.stderr.flush()

    ## Cache-file management to avoid having to parse the ping traces
    def preload(self, file):
        self.links = marshal.loads(open(file).read())

    def postsave(self, file):
        open(file, 'w').write(marshal.dumps(self.links))


if __name__ == "__main__":
    if len(sys.argv) != 4:
        print
        print 'Usage: %s <experiment directory> <cachefile> <trace_time in seconds>' % sys.argv[0]
        print
        print 'Note that the RON tracefile is printed to stdout'
        print
        print 'Example usage:\n\tpython ping2ron.py /home/user/experiments/traces /tmp/cache 10000 > ron_trace'
        print '\n... when the parent directory of the trace distribution is ~user/experiments/'
        raise SystemExit
    cachefile = sys.argv[2]
    p = Ping2Ron(sys.argv[1], int(sys.argv[3]))
    if not os.path.exists(cachefile):
        p.traverse()
        p.postsave(cachefile)
    else:
        p.preload(cachefile)
    p.output_ron()
